<?php

namespace Tests\Feature;

use App\Models\Organization;
use App\Models\OrganizationRole;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TeamManagementTest extends TestCase
{
    use RefreshDatabase;

    private User $owner;
    private User $agent;
    private Organization $organization;

    protected function setUp(): void
    {
        parent::setUp();

        $this->organization = Organization::create([
            "name" => "Test Organization",
            "timezone" => "UTC",
        ]);

        $this->owner = User::create([
            "organization_id" => $this->organization->id,
            "name" => "Test Owner",
            "email" => "owner@test.com",
            "password" => bcrypt("password"),
            "role" => "Owner",
        ]);

        $this->agent = User::create([
            "organization_id" => $this->organization->id,
            "name" => "Test Agent",
            "email" => "agent@test.com",
            "password" => bcrypt("password"),
            "role" => "Agent",
        ]);
    }

    public function test_owner_can_view_team_management_page()
    {
        $response = $this->actingAs($this->owner)->get("/team");
        $response->assertStatus(200);
    }

    public function test_agent_cannot_view_team_management_page()
    {
        $response = $this->actingAs($this->agent)->get("/team");
        $response->assertStatus(403);
    }

    public function test_owner_can_create_custom_role()
    {
        $response = $this->actingAs($this->owner)->post("/team/roles", [
            "name" => "Support Manager",
            "description" => "Can manage contacts and messages",
            "permissions" => ["manage_contacts", "send_messages"],
        ]);

        $response->assertSessionHas("success");
        $this->assertDatabaseHas("organization_roles", [
            "organization_id" => $this->organization->id,
            "name" => "Support Manager",
        ]);
    }

    public function test_owner_can_invite_team_member_with_custom_role()
    {
        $role = OrganizationRole::create([
            "organization_id" => $this->organization->id,
            "name" => "Support Manager",
        ]);

        $response = $this->actingAs($this->owner)->post("/team", [
            "name" => "New Agent",
            "email" => "newagent@test.com",
            "password" => "password123",
            "password_confirmation" => "password123",
            "role" => "Agent",
            "organization_role_id" => $role->id,
        ]);

        $response->assertSessionHas("success");
        $this->assertDatabaseHas("users", [
            "email" => "newagent@test.com",
            "organization_role_id" => $role->id,
        ]);
    }

    public function test_owner_can_update_team_member_role()
    {
        $role = OrganizationRole::create([
            "organization_id" => $this->organization->id,
            "name" => "Support Manager",
        ]);

        $response = $this->actingAs($this->owner)->put("/team/{$this->agent->id}", [
            "name" => "Updated Agent",
            "email" => $this->agent->email,
            "role" => "Agent",
            "organization_role_id" => $role->id,
        ]);

        $response->assertSessionHas("success");
        $this->assertDatabaseHas("users", [
            "id" => $this->agent->id,
            "name" => "Updated Agent",
            "organization_role_id" => $role->id,
        ]);
    }

    public function test_owner_cannot_delete_role_in_use()
    {
        $role = OrganizationRole::create([
            "organization_id" => $this->organization->id,
            "name" => "Support Manager",
        ]);

        $this->agent->update(["organization_role_id" => $role->id]);

        $response = $this->actingAs($this->owner)->delete("/team/roles/{$role->id}");

        $response->assertSessionHasErrors(["role"]);
        $this->assertDatabaseHas("organization_roles", ["id" => $role->id]);
    }
}
