<?php

namespace App\Jobs;

use App\Events\MessageStatusUpdated;
use App\Models\Message;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class ProcessMessageStatus implements ShouldQueue
{
    use Queueable;

    public $wabaId;
    public $statusData;

    /**
     * Create a new job instance.
     */
    public function __construct($wabaId, array $statusData)
    {
        $this->wabaId = $wabaId;
        $this->statusData = $statusData;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $messageId = $this->statusData['id'];
        $status = $this->statusData['status'];
        
        // Find the message by meta_message_id
        $message = Message::withoutGlobalScopes()
            ->where('meta_message_id', $messageId)
            ->first();

        if (!$message) {
            // It's possible we receive a status update for a message we didn't send 
            // (e.g. sent from another system connected to the same WABA), or a race condition.
            Log::info("Received status update for unknown message ID: {$messageId}");
            return;
        }

        // Update status
        // Prioritize: read > delivered > sent
        // If current status is 'read', don't go back to 'delivered'
        if ($this->shouldUpdateStatus($message->status, $status)) {
            $message->update([
                'status' => $status,
                'error_message' => $status === 'failed' ? json_encode($this->statusData['errors'] ?? []) : null,
            ]);

            // Update contact's last_message_at if it's a new status? 
            // Usually not needed for status updates, but good for UI refresh
            
            // Broadcast event
            MessageStatusUpdated::dispatch($message, $message->organization_id);
        }
    }

    protected function shouldUpdateStatus($current, $new)
    {
        $order = [
            'pending' => 0,
            'sent' => 1,
            'delivered' => 2,
            'read' => 3,
            'failed' => 4, 
        ];

        $currentVal = $order[$current] ?? -1;
        $newVal = $order[$new] ?? -1;

        // Special case: failed can override anything except read? 
        // Actually, usually status progresses forward.
        // If we are 'read', we shouldn't go back to 'delivered'.
        
        return $newVal > $currentVal || $new === 'failed';
    }
}
