<?php

namespace App\Jobs;

use App\Models\Contact;
use App\Models\Message;
use App\Models\Organization;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class ProcessInboundMessage implements ShouldQueue
{
    use Queueable;

    public $wabaId;
    public $message;
    public $contactData;
    public $metadata;

    /**
     * Create a new job instance.
     */
    public function __construct($wabaId, array $message, ?array $contactData, ?array $metadata)
    {
        $this->wabaId = $wabaId;
        $this->message = $message;
        $this->contactData = $contactData;
        $this->metadata = $metadata;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // 1. Find Organization
        $organization = Organization::where('waba_id', $this->wabaId)->first();
        
        if (!$organization) {
            Log::warning("Inbound message received for unknown WABA ID: {$this->wabaId}");
            return;
        }

        // Apply tenant scope for safety (though BaseTenantModel usually handles this, 
        // in jobs we sometimes need to explicitly set it or bypass it if we're global)
        // For this job, we'll set the context so subsequent queries use the right org_id
        
        $phoneNumber = $this->message['from'];
        $messageType = $this->message['type'];
        $messageId = $this->message['id'];
        
        // 2. Find or Create Contact
        $contact = Contact::withoutGlobalScopes()
            ->firstOrCreate(
                [
                    'organization_id' => $organization->id,
                    'phone_number' => $phoneNumber
                ],
                [
                    'name' => $this->contactData['profile']['name'] ?? 'Unknown',
                ]
            );

        // 3. Extract Content
        $content = null;
        if ($messageType === 'text') {
            $content = $this->message['text']['body'] ?? '';
            
            // COMPLIANCE: Check for STOP keywords
            $this->checkComplianceKeywords($contact, $content);
        }

        // 4. Save Message
        $inboundMessage = Message::withoutGlobalScopes()->create([
            'organization_id' => $organization->id,
            'contact_id' => $contact->id,
            'meta_message_id' => $messageId,
            'direction' => 'inbound',
            'type' => $messageType,
            'content' => $content,
            'status' => 'delivered',
            'media_metadata' => $this->extractMediaMetadata($messageType, $this->message),
        ]);

        if (!empty($inboundMessage->media_metadata) && $inboundMessage->media_metadata['download_status'] === 'pending') {
            DownloadWhatsAppMedia::dispatch($inboundMessage->id)
                ->onQueue('media_downloads');
        }

        // 5. Trigger Real-time Event (Reverb)
        \App\Events\MessageReceived::dispatch($inboundMessage, $organization->id);
        
        // 6. Handle AI Reply or Routing
        if ($messageType === 'text' && !$contact->opted_out) {
            ProcessAiReply::dispatch($organization->id, $contact->id, $inboundMessage->id)
                ->onQueue('ai_replies');
        }
    }

    /**
     * Critical Logic: The "STOP" Rule
     */
    protected function checkComplianceKeywords(Contact $contact, string $content): void
    {
        $normalizedContent = strtoupper(trim($content));
        $optOutKeywords = ['STOP', 'UNSUBSCRIBE', 'CANCEL'];
        
        if (in_array($normalizedContent, $optOutKeywords)) {
            $contact->update(['opted_out' => true]);
            Log::info("Contact {$contact->phone_number} opted out via keyword: {$normalizedContent}");
        }
    }

    /**
     * Extract media IDs for downloading later via proxy
     */
    protected function extractMediaMetadata(string $type, array $message): ?array
    {
        $mediaTypes = ['image', 'audio', 'video', 'document', 'sticker'];
        
        if (in_array($type, $mediaTypes) && isset($message[$type])) {
            return [
                'id' => $message[$type]['id'],
                'mime_type' => $message[$type]['mime_type'] ?? null,
                'sha256' => $message[$type]['sha256'] ?? null,
                'download_status' => 'pending' // Flag for the media proxy job
            ];
        }
        
        return null;
    }
}
