<?php

namespace App\Jobs;

use App\Models\Campaign;
use App\Services\MetaApiService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class ProcessCampaign implements ShouldQueue
{
    use Queueable;

    public $campaignId;

    /**
     * Create a new job instance.
     */
    public function __construct(int $campaignId)
    {
        $this->campaignId = $campaignId;
    }

    /**
     * Execute the job.
     */
    public function handle(MetaApiService $metaService): void
    {
        $campaign = Campaign::with('contacts')->find($this->campaignId);

        if (!$campaign || $campaign->status !== 'scheduled') {
            return;
        }

        $campaign->update(['status' => 'processing']);
        $organization = $campaign->organization;

        if (!$organization->phone_number_id) {
            Log::error('Campaign Send Error - WhatsApp not connected (missing phone_number_id)', [
                'campaign_id' => $campaign->id,
                'organization_id' => $organization->id,
            ]);
            $campaign->update([
                'status' => 'failed',
            ]);
            return;
        }

        $metaService->setOrganization($organization);

        $successful = 0;
        $failed = 0;

        foreach ($campaign->contacts as $contact) {
            // Frequency Capping: Skip if marketed within last 12 hours
            // TODO: Implement frequency capping if needed using a separate mechanism
            if (false) {
                $campaign->contacts()->updateExistingPivot($contact->id, [
                    'status' => 'failed',
                    'error_message' => 'Frequency cap reached (1 msg / 12 hrs)',
                ]);
                $failed++;
                continue;
            }

            // Skip if opted out
            if ($contact->opted_out) {
                $campaign->contacts()->updateExistingPivot($contact->id, [
                    'status' => 'failed',
                    'error_message' => 'User opted out',
                ]);
                $failed++;
                continue;
            }

            $payload = [
                'messaging_product' => 'whatsapp',
                'to' => $contact->phone_number,
                'type' => 'template',
                'template' => [
                    'name' => $campaign->template_name,
                    'language' => [
                        'code' => $campaign->language,
                    ],
                ]
            ];

            if (!empty($campaign->components)) {
                // Here we would dynamically inject variables per contact (e.g. {{name}})
                // For simplicity, we just pass the static components array
                $payload['template']['components'] = $campaign->components;
            }

            try {
                $response = $metaService->post("/{$organization->phone_number_id}/messages", $payload);

                if (isset($response['messages'][0]['id'])) {
                    $campaign->contacts()->updateExistingPivot($contact->id, [
                        'status' => 'sent',
                    ]);
                    
                    // Update contact stats (TODO: implement if needed)
                    // $contact->update(['last_marketed_at' => now()]);
                    $successful++;
                } else {
                    throw new \Exception('Unknown API response structure');
                }
            } catch (\Exception $e) {
                Log::error('Campaign Send Error', [
                    'campaign_id' => $campaign->id,
                    'contact_id' => $contact->id,
                    'error' => $e->getMessage(),
                ]);

                $campaign->contacts()->updateExistingPivot($contact->id, [
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                ]);
                $failed++;
            }
        }

        $campaign->update([
            'status' => 'completed',
            'successful_sends' => $successful,
            'failed_sends' => $failed,
        ]);
    }
}
