<?php

namespace App\Jobs;

use App\Models\Contact;
use App\Models\Message;
use App\Models\Organization;
use App\Services\Ai\AiGateway;
use App\Services\MetaApiService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class ProcessAiReply implements ShouldQueue
{
    use Queueable;

    public $organizationId;
    public $contactId;
    public $inboundMessageId;

    /**
     * Create a new job instance.
     */
    public function __construct(int $organizationId, int $contactId, int $inboundMessageId)
    {
        $this->organizationId = $organizationId;
        $this->contactId = $contactId;
        $this->inboundMessageId = $inboundMessageId;
    }

    /**
     * Execute the job.
     */
    public function handle(AiGateway $aiGateway, MetaApiService $metaService): void
    {
        $organization = Organization::find($this->organizationId);
        $contact = Contact::withoutGlobalScopes()->find($this->contactId);
        $inboundMessage = Message::withoutGlobalScopes()->find($this->inboundMessageId);

        if (!$organization || !$contact || !$inboundMessage) {
            return;
        }

        // Don't reply if user opted out
        if ($contact->opted_out) {
            return;
        }

        // 1. Fetch conversation history for context (last 10 messages)
        $history = Message::withoutGlobalScopes()
            ->where('contact_id', $contact->id)
            ->where('organization_id', $organization->id)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get()
            ->reverse();

        $messages = [];
        
        // Add system prompt (in a real app, this would be customizable per organization)
        $systemPrompt = $organization->settings['ai_system_prompt'] 
            ?? "You are a helpful customer support assistant for {$organization->name}. Keep your answers concise, professional, and friendly.";
            
        $messages[] = ['role' => 'system', 'content' => $systemPrompt];

        foreach ($history as $msg) {
            if ($msg->type === 'text' && !empty($msg->content)) {
                $messages[] = [
                    'role' => $msg->direction === 'inbound' ? 'user' : 'assistant',
                    'content' => $msg->content,
                ];
            }
        }

        try {
            // 2. Generate AI Reply using the configured driver
            $driver = $aiGateway->driver();
            $replyContent = $driver->chat($messages);

            if (empty($replyContent)) {
                Log::warning("AI returned empty reply for contact {$contact->id}");
                return;
            }

            // 3. Save pending outbound message
            $outboundMessage = Message::withoutGlobalScopes()->create([
                'organization_id' => $organization->id,
                'contact_id' => $contact->id,
                'direction' => 'outbound',
                'type' => 'text',
                'content' => $replyContent,
                'status' => 'pending',
            ]);

            // 4. Send via Meta API
            $metaService->setOrganization($organization);

            if (!$organization->phone_number_id) {
                throw new \Exception('WhatsApp is not connected for this organization (missing phone_number_id)');
            }
            
            $payload = [
                'messaging_product' => 'whatsapp',
                'recipient_type' => 'individual',
                'to' => $contact->phone_number,
                'type' => 'text',
                'text' => [
                    'preview_url' => false,
                    'body' => $replyContent,
                ]
            ];

            $response = $metaService->post("/{$organization->phone_number_id}/messages", $payload);

            if (isset($response['messages'][0]['id'])) {
                $outboundMessage->update([
                    'meta_message_id' => $response['messages'][0]['id'],
                    'status' => 'sent',
                ]);
            }
        } catch (\Exception $e) {
            Log::error('AI Reply Error', [
                'error' => $e->getMessage(),
                'contact_id' => $contact->id,
            ]);
            
            if (isset($outboundMessage)) {
                $outboundMessage->update([
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                ]);
            }
        }
    }
}
