<?php

namespace App\Jobs;

use App\Models\Message;
use App\Models\Organization;
use App\Services\MetaApiService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DownloadWhatsAppMedia implements ShouldQueue
{
    use Queueable;

    public $messageId;

    /**
     * Create a new job instance.
     */
    public function __construct(int $messageId)
    {
        $this->messageId = $messageId;
    }

    /**
     * Execute the job.
     */
    public function handle(MetaApiService $metaService): void
    {
        $message = Message::withoutGlobalScopes()->find($this->messageId);

        if (!$message || empty($message->media_metadata) || $message->media_metadata['download_status'] !== 'pending') {
            return;
        }

        $organization = Organization::find($message->organization_id);
        
        if (!$organization) {
            return;
        }

        $mediaId = $message->media_metadata['id'];
        $metaService->setOrganization($organization);

        try {
            // 1. Get media URL
            $mediaInfo = $metaService->get("/{$mediaId}");
            
            if (!isset($mediaInfo['url'])) {
                throw new \Exception('Media URL not found in Meta response');
            }

            $mediaUrl = $mediaInfo['url'];
            $mimeType = $mediaInfo['mime_type'] ?? 'application/octet-stream';
            
            // Generate a safe filename
            $extension = $this->getExtensionFromMime($mimeType);
            $filename = "waba_media/{$organization->id}/" . Str::uuid() . '.' . $extension;

            // 2. Download the media (Requires passing the Auth token as Bearer)
            $response = Http::withToken($organization->system_user_token)
                ->get($mediaUrl);

            if ($response->failed()) {
                throw new \Exception('Failed to download media file from Meta');
            }

            // 3. Store in configured disk (e.g. s3)
            Storage::disk(config('filesystems.default'))->put($filename, $response->body());

            // 4. Update message metadata
            $updatedMetadata = $message->media_metadata;
            $updatedMetadata['download_status'] = 'completed';
            $updatedMetadata['storage_path'] = $filename;
            $updatedMetadata['url'] = Storage::disk(config('filesystems.default'))->url($filename);

            $message->update(['media_metadata' => $updatedMetadata]);

        } catch (\Exception $e) {
            Log::error('Media Download Error', [
                'message_id' => $message->id,
                'media_id' => $mediaId,
                'error' => $e->getMessage()
            ]);

            $updatedMetadata = $message->media_metadata;
            $updatedMetadata['download_status'] = 'failed';
            $updatedMetadata['error'] = $e->getMessage();
            
            $message->update(['media_metadata' => $updatedMetadata]);
        }
    }

    /**
     * Map common mime types to extensions.
     */
    protected function getExtensionFromMime(string $mime): string
    {
        $map = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/webp' => 'webp',
            'audio/ogg; codecs=opus' => 'ogg',
            'audio/mp4' => 'm4a',
            'audio/mpeg' => 'mp3',
            'video/mp4' => 'mp4',
            'application/pdf' => 'pdf',
        ];

        return $map[$mime] ?? 'bin';
    }
}
