<?php

namespace App\Imports;

use App\Models\Contact;
use App\Models\Group;
use App\Models\Tag;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Illuminate\Support\Facades\DB;

class ContactsImport implements ToCollection, WithHeadingRow, WithChunkReading
{
    protected $organizationId;
    protected $duplicateHandling;
    protected $groupNames;
    protected $tagNames;

    public function __construct($organizationId, $duplicateHandling = 'skip', $groupNames = [], $tagNames = [])
    {
        $this->organizationId = $organizationId;
        $this->duplicateHandling = $duplicateHandling;
        $this->groupNames = is_array($groupNames) ? $groupNames : [];
        $this->tagNames = is_array($tagNames) ? $tagNames : [];
    }

    public function collection(Collection $rows)
    {
        // Get or create groups
        $groupIds = [];
        foreach ($this->groupNames as $groupName) {
            $group = Group::firstOrCreate([
                'organization_id' => $this->organizationId,
                'name' => trim($groupName)
            ]);
            $groupIds[] = $group->id;
        }

        // Get or create tags
        $tagIds = [];
        foreach ($this->tagNames as $tagName) {
            $tag = Tag::firstOrCreate([
                'organization_id' => $this->organizationId,
                'name' => trim($tagName)
            ]);
            $tagIds[] = $tag->id;
        }

        foreach ($rows as $row) {
            if (!isset($row['phone'])) {
                continue; // Phone is required
            }

            $phone = (string)$row['phone'];

            DB::transaction(function () use ($row, $phone, $groupIds, $tagIds) {
                // Determine Opt-out status
                $isOptedOut = isset($row['is_opted_out']) ? filter_var($row['is_opted_out'], FILTER_VALIDATE_BOOLEAN) : false;
                $status = $row['status'] ?? ($isOptedOut ? 'Opted-out' : 'Active');
                if ($status === 'Opted-out') {
                    $isOptedOut = true;
                }

                // Handle custom values from CSV (any column not matching core fields can be considered custom)
                // For simplicity, if there's a specific 'custom_values' JSON string in CSV, we use that.
                // Otherwise, we could dynamically build it. Here we assume standard fields.
                $customValues = isset($row['custom_values']) ? json_decode($row['custom_values'], true) : null;

                $contactData = [
                    'organization_id' => $this->organizationId,
                    'first_name' => $row['first_name'] ?? 'Unknown',
                    'last_name' => $row['last_name'] ?? null,
                    'phone' => $phone,
                    'email' => $row['email'] ?? null,
                    'dob' => $row['dob'] ?? null,
                    'timezone' => $row['timezone'] ?? null,
                    'city' => $row['city'] ?? null,
                    'country' => $row['country'] ?? null,
                    'is_opted_out' => $isOptedOut,
                    'status' => $status,
                    'custom_values' => $customValues,
                ];

                $existingContact = Contact::where('organization_id', $this->organizationId)
                    ->where('phone', $phone)
                    ->first();

                if ($existingContact) {
                    if ($this->duplicateHandling === 'overwrite') {
                        $existingContact->update($contactData);
                        $contact = $existingContact;
                    } elseif ($this->duplicateHandling === 'merge') {
                        $mergedData = array_filter($contactData, fn($val) => !is_null($val));
                        $existingContact->update($mergedData);
                        $contact = $existingContact;
                    } else {
                        // skip
                        return;
                    }
                } else {
                    $contact = Contact::create($contactData);
                }

                // Attach to groups (syncWithoutDetaching to append)
                if (!empty($groupIds)) {
                    $contact->groups()->syncWithoutDetaching($groupIds);
                }

                // Attach tags (syncWithoutDetaching to append)
                if (!empty($tagIds)) {
                    $contact->tags()->syncWithoutDetaching($tagIds);
                }
            });
        }
    }

    public function chunkSize(): int
    {
        return 500;
    }
}
