<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessInboundMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Services\PlatformSettingsService;

class WebhookController extends Controller
{
    /**
     * Handle Meta Webhook Verification
     */
    public function verify(Request $request, PlatformSettingsService $platformSettings)
    {
        $mode = $request->query('hub.mode', $request->query('hub_mode'));
        $token = $request->query('hub.verify_token', $request->query('hub_verify_token'));
        $challenge = $request->query('hub.challenge', $request->query('hub_challenge'));

        $verifyToken = $platformSettings->get('meta.webhook_verify_token', config('meta.webhook_verify_token', 'your_secure_verify_token_here'));

        if ($mode === 'subscribe' && $token === $verifyToken) {
            return response($challenge, 200);
        }

        return response('Forbidden', 403);
    }

    /**
     * Handle incoming Meta Webhooks
     */
    public function handle(Request $request, PlatformSettingsService $platformSettings)
    {
        $appSecret = $platformSettings->get('meta.app_secret', config('meta.app_secret'));
        $signature256 = $request->header('X-Hub-Signature-256');

        if (!$appSecret || !$signature256) {
            Log::warning('Meta Webhook Rejected - Missing signature or app secret', [
                'has_app_secret' => (bool) $appSecret,
                'has_signature_256' => (bool) $signature256,
            ]);
            return response('Forbidden', 403);
        }

        $rawPayload = (string) $request->getContent();
        $expectedSignature = 'sha256=' . hash_hmac('sha256', $rawPayload, $appSecret);

        if (!hash_equals($expectedSignature, $signature256)) {
            Log::warning('Meta Webhook Rejected - Invalid signature', [
                'expected_prefix' => substr($expectedSignature, 0, 20),
                'provided_prefix' => substr((string) $signature256, 0, 20),
            ]);
            return response('Forbidden', 403);
        }

        $payload = json_decode($rawPayload, true) ?? [];

        // Basic validation that it's from WhatsApp
        if (!isset($payload['object']) || $payload['object'] !== 'whatsapp_business_account') {
            return response('Not Found', 404);
        }

        try {
            foreach ($payload['entry'] as $entry) {
                $wabaId = $entry['id'];
                
                foreach ($entry['changes'] as $change) {
                    $value = $change['value'];
                    
                    // Handle Messages
                    if (isset($value['messages'])) {
                        foreach ($value['messages'] as $message) {
                            $contact = $value['contacts'][0] ?? null;
                            $metadata = $value['metadata'] ?? null;
                             
                            // Dispatch job to process message asynchronously
                            ProcessInboundMessage::dispatch($wabaId, $message, $contact, $metadata)
                                ->onQueue('inbound_messages');
                        }
                    }
                    
                    // Handle Status Updates (sent, delivered, read, failed)
                    if (isset($value['statuses'])) {
                        foreach ($value['statuses'] as $status) {
                            \App\Jobs\ProcessMessageStatus::dispatch($wabaId, $status)
                                ->onQueue('message_statuses');
                        }
                    }
                }
            }

            return response('EVENT_RECEIVED', 200);
        } catch (\Exception $e) {
            Log::error('Meta Webhook Processing Error', [
                'error' => $e->getMessage(),
                'payload' => $payload
            ]);
            
            return response('ERROR', 500);
        }
    }
}
