<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\Ticket;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TicketController extends Controller
{
    /**
     * Display a listing of tickets.
     */
    public function index()
    {
        $organization = auth()->user()->organization;

        $tickets = Ticket::where('organization_id', $organization->id)
            ->with(['contact', 'assignee'])
            ->orderBy('created_at', 'desc')
            ->get();

        return Inertia::render('Tickets/Index', [
            'organization' => $organization,
            'tickets' => $tickets,
        ]);
    }

    /**
     * Update the specified ticket.
     */
    public function update(Request $request, Ticket $ticket)
    {
        // Ensure ticket belongs to user's organization
        if ($ticket->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $validated = $request->validate([
            'status' => 'sometimes|string|in:open,resolved,closed',
            'internal_notes' => 'sometimes|string|nullable',
            'assigned_to' => 'sometimes|exists:users,id|nullable',
        ]);

        $ticket->update($validated);

        return back()->with('success', 'Ticket updated successfully.');
    }

    /**
     * Create a ticket from a contact (manual creation).
     */
    public function store(Request $request)
    {
        $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'internal_notes' => 'nullable|string',
        ]);

        $contact = Contact::findOrFail($request->contact_id);

        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        // Check if there's already an open ticket
        $existingTicket = Ticket::where('contact_id', $contact->id)
            ->where('status', 'open')
            ->first();

        if ($existingTicket) {
            return back()->with('error', 'Contact already has an open ticket.');
        }

        Ticket::create([
            'organization_id' => auth()->user()->organization_id,
            'contact_id' => $contact->id,
            'status' => 'open',
            'internal_notes' => $request->internal_notes,
            'assigned_to' => auth()->id(), // Assign to creator by default
        ]);

        return back()->with('success', 'Ticket created successfully.');
    }
}
