<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;

class SettingsController extends Controller
{
    public function index()
    {
        return redirect()->route('settings.general');
    }

    public function updateGeneral(Request $request)
    {
        $organization = auth()->user()->organization;

        $validated = $request->validate([
            'name' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'zip' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'timezone' => 'nullable|string|max:255',
            'enable_notification_sound' => 'nullable|boolean',
            'notification_sound' => 'nullable|string|max:255',
            'contact_fields_location' => 'nullable|string|in:before_address,after_address',
            'custom_fields' => 'nullable|array',
            'enable_ticketing' => 'nullable|boolean',
            'auto_assignment' => 'nullable|string|in:off,auto',
            'reassign_reopened_chats' => 'nullable|boolean',
            'grant_agent_access' => 'nullable|boolean',
            'automation_sequence' => 'nullable|array',
        ]);

        $currentSettings = $organization->settings ?? [];
        
        // Merge the incoming settings fields, preserving existing ones that aren't being updated right now
        $newSettings = array_merge($currentSettings, [
            'address' => $request->has('address') ? ($validated['address'] ?? '') : ($currentSettings['address'] ?? ''),
            'city' => $request->has('city') ? ($validated['city'] ?? '') : ($currentSettings['city'] ?? ''),
            'state' => $request->has('state') ? ($validated['state'] ?? '') : ($currentSettings['state'] ?? ''),
            'zip' => $request->has('zip') ? ($validated['zip'] ?? '') : ($currentSettings['zip'] ?? ''),
            'country' => $request->has('country') ? ($validated['country'] ?? 'United Arab Emirates') : ($currentSettings['country'] ?? 'United Arab Emirates'),
            
            'enable_notification_sound' => $request->has('enable_notification_sound') ? ($validated['enable_notification_sound'] ?? true) : ($currentSettings['enable_notification_sound'] ?? true),
            'notification_sound' => $request->has('notification_sound') ? ($validated['notification_sound'] ?? 'default') : ($currentSettings['notification_sound'] ?? 'default'),
            
            'contact_fields_location' => $request->has('contact_fields_location') ? ($validated['contact_fields_location'] ?? 'before_address') : ($currentSettings['contact_fields_location'] ?? 'before_address'),
            'custom_fields' => $request->has('custom_fields') ? ($validated['custom_fields'] ?? []) : ($currentSettings['custom_fields'] ?? []),

            'enable_ticketing' => $request->has('enable_ticketing') ? ($validated['enable_ticketing'] ?? true) : ($currentSettings['enable_ticketing'] ?? true),
            'auto_assignment' => $request->has('auto_assignment') ? ($validated['auto_assignment'] ?? 'off') : ($currentSettings['auto_assignment'] ?? 'off'),
            'reassign_reopened_chats' => $request->has('reassign_reopened_chats') ? ($validated['reassign_reopened_chats'] ?? false) : ($currentSettings['reassign_reopened_chats'] ?? false),
            'grant_agent_access' => $request->has('grant_agent_access') ? ($validated['grant_agent_access'] ?? false) : ($currentSettings['grant_agent_access'] ?? false),
            
            'automation_sequence' => $request->has('automation_sequence') ? ($validated['automation_sequence'] ?? []) : ($currentSettings['automation_sequence'] ?? []),
        ]);

        $updateData = ['settings' => $newSettings];
        
        if ($request->has('name')) {
            $updateData['name'] = $validated['name'];
        }
        
        if ($request->has('timezone')) {
            $updateData['timezone'] = $validated['timezone'];
        }

        $organization->update($updateData);

        return back()->with('success', 'Settings updated successfully.');
    }

    public function updateManualWhatsappSettings(Request $request)
    {
        $organization = auth()->user()->organization;

        $validated = $request->validate([
            'waba_id' => 'required|string',
            'phone_number_id' => 'required|string',
            'system_user_token' => 'required|string',
        ]);

        $organization->update([
            'waba_id' => $validated['waba_id'],
            'phone_number_id' => $validated['phone_number_id'],
            'system_user_token' => $validated['system_user_token'],
        ]);

        return back()->with('success', 'WhatsApp settings updated manually.');
    }
}
