<?php

namespace App\Http\Controllers;

use App\Models\OrganizationRole;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class OrganizationRoleController extends Controller
{
    /**
     * Store a newly created role in storage.
     */
    public function store(Request $request)
    {
        // Ensure only Tenant Owners or System Admins can manage roles
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage roles.");
        }

        $organization = auth()->user()->organization;

        $validated = $request->validate([
            "name" => [
                "required",
                "string",
                "max:255",
                Rule::unique("organization_roles")->where(function ($query) use ($organization) {
                    return $query->where("organization_id", $organization->id);
                }),
            ],
            "description" => ["nullable", "string", "max:1000"],
            "permissions" => ["nullable", "array"],
        ]);

        $organization->roles()->create($validated);

        return back()->with("success", "Role created successfully.");
    }

    /**
     * Update the specified role in storage.
     */
    public function update(Request $request, OrganizationRole $role)
    {
        // Ensure only Tenant Owners or System Admins can manage roles
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage roles.");
        }

        // Ensure the role belongs to the organization
        if ($role->organization_id !== auth()->user()->organization_id) {
            abort(403, "Unauthorized action. You do not own this role.");
        }

        $validated = $request->validate([
            "name" => [
                "required",
                "string",
                "max:255",
                Rule::unique("organization_roles")->where(function ($query) use ($role) {
                    return $query->where("organization_id", $role->organization_id);
                })->ignore($role->id),
            ],
            "description" => ["nullable", "string", "max:1000"],
            "permissions" => ["nullable", "array"],
        ]);

        $role->update($validated);

        return back()->with("success", "Role updated successfully.");
    }

    /**
     * Remove the specified role from storage.
     */
    public function destroy(OrganizationRole $role)
    {
        // Ensure only Tenant Owners or System Admins can manage roles
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage roles.");
        }

        // Ensure the role belongs to the organization
        if ($role->organization_id !== auth()->user()->organization_id) {
            abort(403, "Unauthorized action. You do not own this role.");
        }

        // Prevent deletion if role is in use
        if ($role->users()->count() > 0) {
            return back()->withErrors(["role" => "Cannot delete role because it is currently assigned to users. Please reassign those users first."]);
        }

        $role->delete();

        return back()->with("success", "Role deleted successfully.");
    }
}
