<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\ContactNote;
use Illuminate\Http\Request;

class ContactNoteController extends Controller
{
    public function index(Contact $contact)
    {
        abort_unless($contact->organization_id === auth()->user()->organization_id, 403);

        $notes = $contact->notes()->with('user:id,name')->get();

        return response()->json(['notes' => $notes]);
    }

    public function store(Request $request, Contact $contact)
    {
        abort_unless($contact->organization_id === auth()->user()->organization_id, 403);

        $validated = $request->validate([
            'content' => 'required|string|max:5000',
        ]);

        $note = ContactNote::create([
            'contact_id' => $contact->id,
            'user_id' => auth()->id(),
            'organization_id' => auth()->user()->organization_id,
            'content' => $validated['content'],
        ]);

        $note->load('user:id,name');

        return response()->json(['note' => $note], 201);
    }

    public function update(Request $request, ContactNote $note)
    {
        abort_unless($note->organization_id === auth()->user()->organization_id, 403);

        $validated = $request->validate([
            'content' => 'sometimes|required|string|max:5000',
            'pinned' => 'sometimes|boolean',
        ]);

        $note->update($validated);
        $note->load('user:id,name');

        return response()->json(['note' => $note]);
    }

    public function destroy(ContactNote $note)
    {
        abort_unless($note->organization_id === auth()->user()->organization_id, 403);

        $note->delete();

        return response()->json(['message' => 'Note deleted.']);
    }
}
