<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\Group;
use App\Models\Tag;
use App\Models\CustomFieldDefinition;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Imports\ContactsImport;
use App\Exports\ContactsExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;

class ContactController extends Controller
{
    public function index(Request $request)
    {
        // Check if user has an organization
        if (!auth()->user()->organization) {
            return Inertia::render('Dashboard/Contacts', [
                'contacts' => collect([]),
                'groups' => collect([]),
                'tags' => collect([]),
                'customFields' => [],
                'filters' => $request->only(['search', 'groups', 'tags', 'status', 'sort_by', 'sort_direction']),
                'visibleColumns' => ['first_name', 'phone_number', 'email', 'status'],
                'allColumns' => [
                    ['key' => 'first_name', 'label' => 'Name'],
                    ['key' => 'phone_number', 'label' => 'Phone'],
                    ['key' => 'email', 'label' => 'Email'],
                    ['key' => 'city', 'label' => 'City'],
                    ['key' => 'country', 'label' => 'Country'],
                    ['key' => 'gender', 'label' => 'Gender'],
                    ['key' => 'dob', 'label' => 'Date of Birth'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'groups', 'label' => 'Groups'],
                    ['key' => 'tags', 'label' => 'Tags'],
                ],
                'canManageCustomFields' => false,
            ]);
        }

        $orgId = auth()->user()->organization_id;

        $stats = [
            'total' => Contact::where('organization_id', $orgId)->count(),
            'active' => Contact::where('organization_id', $orgId)->where('status', 'Active')->count(),
            'opted_out' => Contact::where('organization_id', $orgId)->whereIn('status', ['Opted-out', 'Opted_out'])->count(),
        ];

        $sortBy = $request->input('sort_by', 'first_name');
        $sortDirection = $request->input('sort_direction', 'asc');

        // Allowed sort fields to prevent SQL injection
        $allowedSortFields = ['first_name', 'last_name', 'phone', 'phone_number', 'email', 'city', 'country', 'gender', 'dob', 'status', 'created_at'];
        if (!in_array($sortBy, $allowedSortFields)) {
            $sortBy = 'first_name';
        }

        $query = Contact::with(['groups', 'tags'])
            ->where('organization_id', auth()->user()->organization_id)
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                        ->orWhere('last_name', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%")
                        ->orWhere('phone_number', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                });
            })
            ->when($request->groups, function ($query, $groups) {
                if (is_array($groups)) {
                    $query->whereHas('groups', function ($q) use ($groups) {
                        $q->whereIn('groups.id', $groups);
                    });
                }
            })
            ->when($request->tags, function ($query, $tags) {
                if (is_array($tags)) {
                    $query->whereHas('tags', function ($q) use ($tags) {
                        $q->whereIn('tags.id', $tags);
                    });
                }
            })
            ->when($request->status, function ($query, $status) {
                $query->where('status', $status);
            })
            ->orderBy($sortBy, $sortDirection);

        $perPage = in_array((int) $request->input('per_page'), [25, 50, 100]) ? (int) $request->input('per_page') : 50;
        $contacts = $query->paginate($perPage)->withQueryString();
        
        $groups = Group::where('organization_id', auth()->user()->organization_id)
            ->select('id', 'name')->withCount('contacts')->get();
        $tags = Tag::where('organization_id', auth()->user()->organization_id)
            ->select('id', 'name', 'color')->withCount('contacts')->get();

        $customFields = CustomFieldDefinition::where('organization_id', auth()->user()->organization_id)
            ->ordered()
            ->get()
            ->map(function ($field) {
                return [
                    'key' => $field->key,
                    'label' => $field->label,
                    'type' => $field->type,
                    'required' => $field->required,
                    'defaultVisible' => $field->default_visible,
                    'options' => $field->options,
                    'default_value' => $field->default_value,
                ];
            });

        // Get user's column preferences or use defaults
        $visibleColumns = $request->user()->column_preferences ?? ['first_name', 'phone_number', 'status'];

        $allColumns = [
            ['key' => 'first_name', 'label' => 'Name'],
            ['key' => 'phone_number', 'label' => 'Phone'],
            ['key' => 'address', 'label' => 'Address'],
            ['key' => 'city', 'label' => 'City'],
            ['key' => 'country', 'label' => 'Country'],
            ['key' => 'po_box', 'label' => 'PO Box'],
            ['key' => 'gender', 'label' => 'Gender'],
            ['key' => 'dob', 'label' => 'Date of Birth'],
            ['key' => 'status', 'label' => 'Status'],
            ['key' => 'groups', 'label' => 'Groups'],
            ['key' => 'tags', 'label' => 'Tags'],
        ];

        // Add custom fields to columns
        foreach ($customFields as $field) {
            $allColumns[] = [
                'key' => 'custom_' . $field['key'],
                'label' => $field['label'],
                'custom_field_key' => $field['key'],
                'is_custom' => true
            ];
        }

        return Inertia::render('Dashboard/Contacts', [
            'contacts' => $contacts,
            'groups' => $groups,
            'tags' => $tags,
            'customFields' => $customFields,
            'stats' => $stats,
            'filters' => $request->only(['search', 'groups', 'tags', 'status', 'sort_by', 'sort_direction', 'per_page']),
            'visibleColumns' => $visibleColumns,
            'allColumns' => $allColumns,
            'canManageCustomFields' => auth()->user()->canManageCustomFields(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'phone' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'timezone' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:1000',
            'city' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'po_box' => 'nullable|string|max:50',
            'is_opted_out' => 'boolean',
            'custom_values' => 'nullable|array',
            'status' => 'required|in:Active,Opted-out',
            'group_ids' => 'nullable|array',
            'group_ids.*' => 'exists:groups,id',
            'tag_ids' => 'nullable|array',
            'tag_ids.*' => 'exists:tags,id',
        ]);

        // phone must be unique within organization
        $existing = Contact::where('organization_id', auth()->user()->organization_id)
            ->where('phone', $validated['phone'])
            ->first();

        if ($existing) {
            return back()->withErrors(['phone' => 'A contact with this phone number already exists.']);
        }

        DB::transaction(function () use ($validated) {
            $validated['organization_id'] = auth()->user()->organization_id;
            $validated['phone_number'] = $validated['phone'];
            
            // Sync status with is_opted_out if it's not explicitly set or to keep them aligned
            if (isset($validated['is_opted_out']) && $validated['is_opted_out']) {
                $validated['status'] = 'Opted-out';
            } elseif ($validated['status'] === 'Opted-out') {
                $validated['is_opted_out'] = true;
            } else {
                $validated['is_opted_out'] = false;
            }

            $contact = Contact::create($validated);

            if (!empty($validated['group_ids'])) {
                $contact->groups()->sync($validated['group_ids']);
            }

            if (!empty($validated['tag_ids'])) {
                $contact->tags()->sync($validated['tag_ids']);
            }
        });

        return back()->with('success', 'Contact created successfully.');
    }

    public function update(Request $request, Contact $contact)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'phone' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'timezone' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:1000',
            'city' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'po_box' => 'nullable|string|max:50',
            'is_opted_out' => 'boolean',
            'custom_values' => 'nullable|array',
            'status' => 'required|in:Active,Opted-out',
            'group_ids' => 'nullable|array',
            'group_ids.*' => 'exists:groups,id',
            'tag_ids' => 'nullable|array',
            'tag_ids.*' => 'exists:tags,id',
        ]);

        $existing = Contact::where('organization_id', auth()->user()->organization_id)
            ->where('phone', $validated['phone'])
            ->where('id', '!=', $contact->id)
            ->first();

        if ($existing) {
            return back()->withErrors(['phone' => 'A contact with this phone number already exists.']);
        }

        DB::transaction(function () use ($contact, $validated) {
            // Sync status with is_opted_out
            if (isset($validated['is_opted_out']) && $validated['is_opted_out']) {
                $validated['status'] = 'Opted-out';
            } elseif ($validated['status'] === 'Opted-out') {
                $validated['is_opted_out'] = true;
            } else {
                $validated['is_opted_out'] = false;
            }

            $validated['phone_number'] = $validated['phone'];
            $contact->update($validated);

            if (isset($validated['group_ids'])) {
                $contact->groups()->sync($validated['group_ids']);
            } else {
                $contact->groups()->detach();
            }

            if (isset($validated['tag_ids'])) {
                $contact->tags()->sync($validated['tag_ids']);
            } else {
                $contact->tags()->detach();
            }
        });

        return back()->with('success', 'Contact updated successfully.');
    }

    public function destroy(Contact $contact)
    {
        $contact->delete();
        return back()->with('success', 'Contact deleted successfully.');
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id'
        ]);

        Contact::where('organization_id', auth()->user()->organization_id)
            ->whereIn('id', $request->contact_ids)->delete();

        return back()->with('success', 'Selected contacts deleted successfully.');
    }

    public function bulkGroup(Request $request)
    {
        $request->validate([
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contacts,id',
            'group_ids' => 'required|array',
            'group_ids.*' => 'exists:groups,id'
        ]);

        $contacts = Contact::whereIn('id', $request->contact_ids)->get();
        
        foreach ($contacts as $contact) {
            // syncWithoutDetaching to append groups without removing existing ones
            $contact->groups()->syncWithoutDetaching($request->group_ids);
        }

        return back()->with('success', 'Groups applied to selected contacts.');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:csv,xlsx,xls',
            'duplicate_handling' => 'required|in:skip,overwrite,merge',
            'groups' => 'nullable|array',
            'tags' => 'nullable|array'
        ]);

        try {
            Excel::import(
                new ContactsImport(
                    auth()->user()->organization_id, 
                    $request->duplicate_handling,
                    $request->groups ?? [],
                    $request->tags ?? []
                ), 
                $request->file('file')
            );
            return back()->with('success', 'Contacts imported successfully.');
        } catch (\Exception $e) {
            return back()->withErrors(['file' => 'Error importing file: ' . $e->getMessage()]);
        }
    }

    public function saveColumnPreferences(Request $request)
    {
        $validated = $request->validate([
            'visible_columns' => 'required|array',
            'visible_columns.*' => 'string'
        ]);

        $user = auth()->user();
        $user->column_preferences = $validated['visible_columns'];
        $user->save();

        return back()->with('success', 'Column preferences saved successfully.');
    }

    public function export(Request $request)
    {
        $query = Contact::with(['groups', 'tags'])
            ->where('organization_id', auth()->user()->organization_id)
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                        ->orWhere('last_name', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->when($request->group_id, function ($query, $groupId) {
                $query->whereHas('groups', function ($q) use ($groupId) {
                    $q->where('groups.id', $groupId);
                });
            })
            ->when($request->tag_id, function ($query, $tagId) {
                $query->whereHas('tags', function ($q) use ($tagId) {
                    $q->where('tags.id', $tagId);
                });
            });

        return Excel::download(new ContactsExport($query), 'contacts_export_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    public function updateCustomFields(Request $request)
    {
        abort_unless(auth()->user()->canManageCustomFields(), 403);

        $validated = $request->validate([
            'fields' => 'array',
            'fields.*.key' => 'required|string|max:190',
            'fields.*.label' => 'required|string|max:190',
            'fields.*.type' => 'required|string|in:text,textarea,number,date,select,checkbox,radio,email,url',
            'fields.*.required' => 'boolean',
            'fields.*.defaultVisible' => 'boolean',
            'fields.*.options' => 'nullable|array',
            'fields.*.options.*' => 'string|max:190',
            'fields.*.default_value' => 'nullable|string|max:190',
        ]);

        $organizationId = auth()->user()->organization_id;
        
        DB::transaction(function () use ($validated, $organizationId) {
            // Get existing fields
            $existingFieldIds = CustomFieldDefinition::where('organization_id', $organizationId)
                ->pluck('id')
                ->toArray();

            $processedFieldIds = [];

            foreach ($validated['fields'] as $index => $fieldData) {
                $fieldData['organization_id'] = $organizationId;
                $fieldData['default_visible'] = $fieldData['defaultVisible'];
                $fieldData['sort_order'] = $index;

                // Remove frontend-only fields
                unset($fieldData['defaultVisible']);

                if (isset($fieldData['id']) && str_starts_with($fieldData['id'], 'temp_')) {
                    // New field (temporary ID)
                    unset($fieldData['id']);
                }

                if (isset($fieldData['id'])) {
                    // Existing field by ID
                    $field = CustomFieldDefinition::find($fieldData['id']);
                    
                    if ($field && $field->organization_id === $organizationId) {
                        $field->update($fieldData);
                    } else {
                        // Fallback to updateOrCreate if ID invalid/not found/wrong org (shouldn't happen with valid UI)
                         $field = CustomFieldDefinition::updateOrCreate(
                            ['organization_id' => $organizationId, 'key' => $fieldData['key']],
                            $fieldData
                        );
                    }
                } else {
                    // New field without ID - check if key exists to avoid duplicates
                    $field = CustomFieldDefinition::updateOrCreate(
                        ['organization_id' => $organizationId, 'key' => $fieldData['key']],
                        $fieldData
                    );
                }
                
                $processedFieldIds[] = $field->id;
            }

            // Delete fields that are no longer present
            $fieldsToDelete = array_diff($existingFieldIds, $processedFieldIds);
            if (!empty($fieldsToDelete)) {
                CustomFieldDefinition::whereIn('id', $fieldsToDelete)->delete();
            }
        });

        return back()->with('success', 'Custom fields updated successfully.');
    }
}
