<?php

namespace App\Http\Controllers;

use App\Models\CannedResponse;
use App\Models\Contact;
use App\Models\Message;
use App\Models\User;
use App\Services\MetaApiService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class ChatController extends Controller
{
    /**
     * Show the unified chat inbox.
     */
    public function index(Request $request)
    {
        $organization = auth()->user()->organization;

        // if (!$organization->phone_number_id) {
        //     return redirect()->route('settings.whatsapp')
        //         ->with('error', 'WhatsApp is not connected. Please connect WhatsApp first.');
        // }

        $contacts = Contact::where('organization_id', $organization->id)
            ->whereHas('messages')
            ->with(['tags', 'groups', 'latestMessage', 'assignee:id,name,email,role'])
            ->orderByDesc('last_message_at')
            ->get()
            ->map(function ($contact) {
                $contact->display_name = trim(($contact->first_name ?? '') . ' ' . ($contact->last_name ?? '')) ?: ($contact->name ?? $contact->phone_number);
                return $contact;
            });

        $teamMembers = User::where('organization_id', $organization->id)
            ->where('status', '!=', 'inactive')
            ->select('id', 'name', 'email', 'role')
            ->get();

        $cannedResponses = CannedResponse::where('organization_id', $organization->id)->get();

        return Inertia::render('Chat/Index', [
            'organization' => $organization,
            'initialContacts' => $contacts,
            'teamMembers' => $teamMembers,
            'cannedResponses' => $cannedResponses,
        ]);
    }

    /**
     * Get messages for a specific contact (paginated).
     */
    public function getMessages(Contact $contact, Request $request)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $query = Message::where('contact_id', $contact->id)
            ->orderByDesc('created_at');

        if ($request->has('before_id')) {
            $query->where('id', '<', $request->before_id);
        }

        $messages = $query->limit(50)->get()->reverse()->values();

        // Mark unread messages as read
        if ($contact->unread_count > 0) {
            $contact->update(['unread_count' => 0]);
        }

        return response()->json([
            'messages' => $messages,
            'has_more' => $messages->count() === 50,
            'contact' => $contact->load(['tags', 'groups', 'assignee:id,name,email,role']),
        ]);
    }

    /**
     * Get full contact details for the right panel.
     */
    public function getContactDetails(Contact $contact)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $contact->load(['tags', 'groups', 'assignee:id,name,email,role', 'notes']);
        $contact->display_name = trim(($contact->first_name ?? '') . ' ' . ($contact->last_name ?? '')) ?: ($contact->name ?? $contact->phone_number);

        return response()->json($contact);
    }

    /**
     * Send a text message to a contact.
     */
    public function sendMessage(Request $request, Contact $contact, MetaApiService $metaService)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        if ($contact->opted_out) {
            return response()->json(['error' => 'Contact has opted out of messaging.'], 400);
        }

        $request->validate([
            'type' => 'required|string|in:text,image,video,audio,document,location,template',
            'content' => 'nullable|string',
            'media_url' => 'nullable|string',
            'media_file' => 'nullable|file|max:16384',
            'filename' => 'nullable|string',
            'caption' => 'nullable|string',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'location_name' => 'nullable|string',
            'location_address' => 'nullable|string',
            'template_name' => 'nullable|string',
            'template_language' => 'nullable|string',
            'template_components' => 'nullable|array',
        ]);

        $organization = auth()->user()->organization;

        if (!$organization->phone_number_id) {
            return response()->json(['error' => 'WhatsApp is not connected.'], 400);
        }

        $type = $request->type;
        $content = $request->content;
        $mediaMetadata = null;

        // Build the WhatsApp API payload based on message type
        $payload = [
            'messaging_product' => 'whatsapp',
            'recipient_type' => 'individual',
            'to' => $contact->phone_number,
            'type' => $type,
        ];

        switch ($type) {
            case 'text':
                $payload['text'] = [
                    'preview_url' => true,
                    'body' => $content,
                ];
                break;

            case 'image':
            case 'video':
            case 'audio':
            case 'document':
                $mediaData = [];
                if ($request->hasFile('media_file')) {
                    $file = $request->file('media_file');
                    $path = $file->store("chat-media/{$organization->id}", 'public');
                    $mediaData['link'] = url(Storage::url($path));
                    $mediaMetadata = [
                        'local_path' => $path,
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                        'original_name' => $file->getClientOriginalName(),
                    ];
                } elseif ($request->media_url) {
                    $mediaData['link'] = $request->media_url;
                }

                if ($request->caption && in_array($type, ['image', 'video', 'document'])) {
                    $mediaData['caption'] = $request->caption;
                    $content = $request->caption;
                }
                if ($type === 'document' && $request->filename) {
                    $mediaData['filename'] = $request->filename;
                }
                $payload[$type] = $mediaData;
                break;

            case 'location':
                $payload['location'] = [
                    'longitude' => $request->longitude,
                    'latitude' => $request->latitude,
                    'name' => $request->location_name ?? '',
                    'address' => $request->location_address ?? '',
                ];
                $content = ($request->location_name ?? 'Location') . ': ' . $request->latitude . ', ' . $request->longitude;
                break;

            case 'template':
                $payload['template'] = [
                    'name' => $request->template_name,
                    'language' => ['code' => $request->template_language ?? 'en'],
                ];
                if ($request->template_components) {
                    $payload['template']['components'] = $request->template_components;
                }
                $content = "[Template: {$request->template_name}]";
                break;
        }

        // Save pending message to database
        $message = Message::create([
            'organization_id' => $organization->id,
            'contact_id' => $contact->id,
            'direction' => 'outbound',
            'type' => $type,
            'content' => $content,
            'status' => 'pending',
            'media_metadata' => $mediaMetadata,
        ]);

        // Send via Meta API
        try {
            $metaService->setOrganization($organization);
            $response = $metaService->post("/{$organization->phone_number_id}/messages", $payload);

            if (isset($response['messages'][0]['id'])) {
                $message->update([
                    'meta_message_id' => $response['messages'][0]['id'],
                    'status' => 'sent',
                ]);
            }

            // Update contact's last message time
            $contact->update([
                'last_message_at' => now(),
                'conversation_status' => $contact->conversation_status === 'new' ? 'open' : $contact->conversation_status,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send WhatsApp message', [
                'contact_id' => $contact->id,
                'error' => $e->getMessage(),
            ]);

            $message->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return response()->json([
                'error' => 'Failed to send message: ' . $e->getMessage(),
                'message' => $message->fresh(),
            ], 500);
        }

        return response()->json($message->fresh());
    }

    /**
     * Upload media file and return the URL (for previewing before send).
     */
    public function uploadMedia(Request $request)
    {
        $request->validate([
            'file' => 'required|file|max:16384',
        ]);

        $organization = auth()->user()->organization;
        $file = $request->file('file');
        $path = $file->store("chat-media/{$organization->id}", 'public');

        return response()->json([
            'url' => url(Storage::url($path)),
            'path' => $path,
            'name' => $file->getClientOriginalName(),
            'size' => $file->getSize(),
            'mime_type' => $file->getMimeType(),
        ]);
    }

    /**
     * Assign a contact/conversation to a team member.
     */
    public function assignAgent(Request $request, Contact $contact)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $request->validate([
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        // Verify the assigned user belongs to the same organization
        if ($request->assigned_to) {
            $agent = User::where('id', $request->assigned_to)
                ->where('organization_id', auth()->user()->organization_id)
                ->firstOrFail();
        }

        $contact->update([
            'assigned_to' => $request->assigned_to,
            'conversation_status' => $request->assigned_to ? 'open' : $contact->conversation_status,
        ]);

        return response()->json([
            'success' => true,
            'contact' => $contact->load('assignee:id,name,email,role'),
        ]);
    }

    /**
     * Update conversation status.
     */
    public function updateStatus(Request $request, Contact $contact)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $request->validate([
            'status' => 'required|in:new,open,pending,resolved,closed',
        ]);

        $contact->update(['conversation_status' => $request->status]);

        return response()->json(['success' => true, 'contact' => $contact]);
    }

    /**
     * Toggle AI assistance for a conversation.
     */
    public function toggleAi(Request $request, Contact $contact)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $contact->update(['ai_enabled' => !$contact->ai_enabled]);

        return response()->json([
            'success' => true,
            'ai_enabled' => $contact->ai_enabled,
        ]);
    }

    /**
     * Update internal note for a contact.
     */
    public function updateNote(Request $request, Contact $contact)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $request->validate(['note' => 'nullable|string|max:5000']);

        $contact->update(['internal_note' => $request->note]);

        return response()->json(['success' => true]);
    }

    /**
     * Fetch message templates from Meta API.
     */
    public function getTemplates(MetaApiService $metaService)
    {
        $organization = auth()->user()->organization;

        if (!$organization->waba_id) {
            return response()->json(['templates' => []]);
        }

        try {
            $metaService->setOrganization($organization);
            $response = $metaService->get("/{$organization->waba_id}/message_templates", [
                'fields' => 'name,status,language,category,components',
                'limit' => 100,
            ]);

            $templates = collect($response['data'] ?? [])
                ->where('status', 'APPROVED')
                ->values();

            return response()->json(['templates' => $templates]);
        } catch (\Exception $e) {
            Log::error('Failed to fetch templates', ['error' => $e->getMessage()]);
            return response()->json(['templates' => [], 'error' => $e->getMessage()]);
        }
    }

    /**
     * Search conversations.
     */
    public function searchConversations(Request $request)
    {
        $query = $request->get('q', '');
        $orgId = auth()->user()->organization_id;

        if (strlen($query) < 2) {
            return response()->json([]);
        }

        $contacts = Contact::where('organization_id', $orgId)
            ->whereHas('messages')
            ->where(function ($q) use ($query) {
                $q->where('first_name', 'like', "%{$query}%")
                  ->orWhere('last_name', 'like', "%{$query}%")
                  ->orWhere('name', 'like', "%{$query}%")
                  ->orWhere('phone_number', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->with(['latestMessage', 'assignee:id,name'])
            ->orderByDesc('last_message_at')
            ->limit(20)
            ->get()
            ->map(function ($contact) {
                $contact->display_name = trim(($contact->first_name ?? '') . ' ' . ($contact->last_name ?? '')) ?: ($contact->name ?? $contact->phone_number);
                return $contact;
            });

        return response()->json($contacts);
    }

    /**
     * CRUD for canned responses.
     */
    public function getCannedResponses()
    {
        $responses = CannedResponse::where('organization_id', auth()->user()->organization_id)->get();
        return response()->json($responses);
    }

    public function storeCannedResponse(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:100',
            'content' => 'required|string|max:2000',
            'shortcut' => 'nullable|string|max:50',
            'category' => 'nullable|string|max:50',
        ]);

        $response = CannedResponse::create([
            'organization_id' => auth()->user()->organization_id,
            'title' => $request->title,
            'content' => $request->content,
            'shortcut' => $request->shortcut,
            'category' => $request->category,
        ]);

        return response()->json($response, 201);
    }

    public function deleteCannedResponse(CannedResponse $cannedResponse)
    {
        if ($cannedResponse->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $cannedResponse->delete();
        return response()->json(['success' => true]);
    }

    /**
     * Mark messages as read (WhatsApp read receipts).
     */
    public function markAsRead(Contact $contact, MetaApiService $metaService)
    {
        if ($contact->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $organization = auth()->user()->organization;

        // Get latest inbound message to send read receipt
        $lastInbound = Message::where('contact_id', $contact->id)
            ->where('direction', 'inbound')
            ->latest()
            ->first();

        if ($lastInbound && $lastInbound->meta_message_id && $organization->phone_number_id) {
            try {
                $metaService->setOrganization($organization);
                $metaService->post("/{$organization->phone_number_id}/messages", [
                    'messaging_product' => 'whatsapp',
                    'status' => 'read',
                    'message_id' => $lastInbound->meta_message_id,
                ]);
            } catch (\Exception $e) {
                // Silently fail - read receipts are not critical
                Log::warning('Failed to send read receipt', ['error' => $e->getMessage()]);
            }
        }

        $contact->update(['unread_count' => 0]);

        return response()->json(['success' => true]);
    }
}
