<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Validation\Rule;

class TeamController extends Controller
{
    /**
     * Display the team management dashboard for super admins.
     */
    public function index()
    {
        // Ensure only System Admins can access this page
        if (auth()->user()->role !== 'System_Admin') {
            abort(403, 'Unauthorized action. Only System Admins can manage the global team.');
        }

        $team = User::whereNull('organization_id')
            ->where('role', 'System_Admin')
            ->orderBy('created_at', 'desc')
            ->get(['id', 'name', 'email', 'role', 'created_at']);

        return Inertia::render('Admin/Team', [
            'team' => $team,
        ]);
    }

    /**
     * Store a newly created system admin.
     */
    public function store(Request $request)
    {
        // Ensure only System Admins can create other system admins
        if (auth()->user()->role !== 'System_Admin') {
            abort(403, 'Unauthorized action. Only System Admins can manage the global team.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        User::create([
            'organization_id' => null, // System Admins don't belong to a tenant
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'System_Admin',
        ]);

        return back()->with('success', 'System Admin created successfully.');
    }

    /**
     * Update the specified system admin details.
     */
    public function update(Request $request, User $user)
    {
        // Ensure only System Admins can update team members
        if (auth()->user()->role !== 'System_Admin') {
            abort(403, 'Unauthorized action. Only System Admins can manage the global team.');
        }

        // Ensure the target user is a System Admin
        if ($user->role !== 'System_Admin' || !is_null($user->organization_id)) {
            abort(403, 'Unauthorized action. You can only edit System Admins here.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => ['nullable', 'string', 'min:8', 'confirmed'],
        ]);

        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
        ];

        if (!empty($validated['password'])) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        $user->update($updateData);

        return back()->with('success', 'System Admin updated successfully.');
    }

    /**
     * Remove the specified system admin.
     */
    public function destroy(User $user)
    {
        // Ensure only System Admins can delete team members
        if (auth()->user()->role !== 'System_Admin') {
            abort(403, 'Unauthorized action. Only System Admins can manage the global team.');
        }

        // Ensure the target user is a System Admin
        if ($user->role !== 'System_Admin' || !is_null($user->organization_id)) {
            abort(403, 'Unauthorized action. You can only delete System Admins here.');
        }

        // Prevent self-deletion
        if ($user->id === auth()->user()->id) {
            abort(403, 'Unauthorized action. You cannot delete your own account.');
        }

        $user->delete();

        return back()->with('success', 'System Admin removed successfully.');
    }
}
